-----------------------------------------------------------------
--
--  Pico_test.vhd
--
--  Author: Nial Stewart, Nial Stewart Developments Ltd.
--          www.nialstewart.co.uk
--          Jan '03
--
-- This is a simple wrapper round an instance of the Xilinx 
-- Picoblaze micro-controller see...
-- http://www.xilinx.com/ipcenter/processor_central/picoblaze/index.htm
--
-- This is a very simple example of what the picoblaze is capable
-- of and is designed as a 'getting' started example for
-- use with the BurchED fpga evaluation boards.
--
-----------------------------------------------------------------


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all; 
use ieee.std_logic_unsigned.all;


entity pico_test is
port (clk       : in std_logic;             -- Clk (80MHz in this example)

      intrpt_n  : in std_logic;             -- Interrupt input (from button on board), '0' when..
                                            -- ..the switch is pressed, '1' when released.
      test_led  : out std_logic;            -- Output to single Test LED on BurchED FPGA board.
    
      leds      : out std_logic_vector(7 downto 0) -- Output to BurchED B5-LEDs module
      );
end pico_test;




architecture rtl of pico_test is


-----------------------------------------------------------------
--     Component declatations (not needed with Vhdl'93)
-----------------------------------------------------------------


-- This is the actual Picoblaze processor (KCPSM was the old name)

component kcpsm
port (address       : out std_logic_vector(7 downto 0);
      instruction   : in std_logic_vector(15 downto 0);
      port_id       : out std_logic_vector(7 downto 0);
      write_strobe  : out std_logic;
      out_port      : out std_logic_vector(7 downto 0);
      read_strobe   : out std_logic;
      in_port       : in std_logic_vector(7 downto 0);
      interrupt     : in std_logic;
      reset         : in std_logic;
      clk           : in std_logic);
end component;


-- This is what the assembler generates, it's a wrapper for a RAM
-- that's initialised to hold the firmware.

component picocode is
port (address     : in std_logic_vector(7 downto 0);
      instruction : out std_logic_vector(15 downto 0);
      clk         : in std_logic);
end component;





-----------------------------------------------------------------
--     Signal declarations
-----------------------------------------------------------------

signal reset       : std_logic;
signal rst_n       : std_logic;
signal reset_count : std_logic_vector(9 downto 0);

signal address     : std_logic_vector(7 downto 0);
signal instruction : std_logic_vector(15 downto 0);

signal port_id     : std_logic_vector(7 downto 0);
signal wr          : std_logic;
signal out_port    : std_logic_vector(7 downto 0);
signal rd          : std_logic;
signal in_port     : std_logic_vector(7 downto 0);


-- Following used to detect rising edge and generate an interrupt
signal int_in      : std_logic;
signal int_d,int_2d : std_logic;
signal fall_edge   : std_logic;
signal fall_edge_d : std_logic;
signal interrupt   : std_logic;



begin

-----------------------------------------------------------------
-- Reset generation. This isn't very robust, but will suffice
-- for this demonstration.
-- After configuration all signals are set to '0' (unless 
-- specified otherwise in the Xilinx tools). To get an active
-- high reset signal we need to invert a signal which powers
-- up to the default '0'. After a delay of 1023 clocks invert
-- this signal and de-activate the reset signal.
-----------------------------------------------------------------

process(clk)
begin
if(rising_edge(clk)) then
  reset_count <= reset_count + '1';
  if(reset_count = "1111111111") then
    rst_n <= '1';
  end if;
  
  reset <= not(rst_n);
end if;
end process;


-----------------------------------------------------------------
-- Register the interrupt in and detect a button press (ie a 
-- falling edge on the input.
-----------------------------------------------------------------

process(clk,reset)
begin
if(reset = '1') then
  int_in <= '0';
  interrupt <= '0';
  int_d <= '0';
  int_2d <= '0';
elsif(rising_edge(clk)) then
  int_in <= intrpt_n;                 -- Register in..
  int_d <= int_in;                  -- ..register again...
  int_2d <= int_d;                  -- .. then again to detect rising edge.
  
  fall_edge <= (not int_d) and int_2d;  -- The int pulse must be 2 clock cycles
  fall_edge_d <= fall_edge;              -- wide to guarantee it's detected.
  
  interrupt <= fall_edge or fall_edge_d;
  
end if;
end process;



-----------------------------------------------------------------
-- Process to steer port output to the correct destination
-----------------------------------------------------------------

process(reset,clk)
begin
if(reset = '1') then
  test_led <= '0';
  leds     <= (others => '0');
elsif(rising_edge(clk)) then
  if(wr = '1') then
    case port_id is
      when "00000010" =>
        test_led <= out_port(0);      
      when others =>
        leds  <= out_port;
    end case;
  end if;
end if;
end process;





pico_i:kcpsm
port map (address       => address, -- out std_logic_vector(7 downto 0);
          instruction   => instruction, -- in std_logic_vector(15 downto 0);
          port_id       => port_id, -- out std_logic_vector(7 downto 0);
          write_strobe  => wr, -- out std_logic;
          out_port      => out_port, -- out std_logic_vector(7 downto 0);
          read_strobe   => rd, -- out std_logic;
          in_port       => in_port, -- in std_logic_vector(7 downto 0);
          interrupt     => interrupt, -- in std_logic;
          reset         => reset, -- in std_logic;
          clk           => clk -- in std_logic)
          );

code_i:picocode 
port map(address      => address, -- in std_logic_vector(7 downto 0);
        instruction   => instruction, -- out std_logic_vector(15 downto 0);
        clk           => clk -- in std_logic);
        );





end rtl;
