-----------------------------------------------------------------
--
--  chip.vhd
--
--  Author: Nial Stewart, Nial Stewart Developments Ltd.
--          www.nialstewart.co.uk
--
--  Top level Chip 'wrapper' to demonstrate the USB/PC interface.
--  The main function of this wrapper is to implement the top
--  level data bus tri-state. This can be buried down the hierarchy
--  of the design but for ease of synthesis is kept at the top level
--  (some synth tools need switches to bubble tri-states up the hierarchy
--  to the top level).
--  
--  This also connects the reg outputs to the state inputs for this
--  example to stop everything that doesn't go anywhere being
--  synthesised out by the tools.
--
-----------------------------------------------------------------


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;  


entity chip is
port (rst_n     : in std_logic;           -- Reset
    clk       : in std_logic;             -- Clk (80MHz in this example)
    rxf_n     : in std_logic;             -- Receive fifo flag
    txe_n     : in std_logic;             -- Tranmit fifo flag
    data      : inout std_logic_vector(7 downto 0); -- Bi-directional data bus
    wr        : out std_logic;            -- Write signal to FT245BM
    rd_n      : out std_logic;            -- Read    "
                                                      -- Next two signals aren't used
    pwren_n   : in std_logic;             -- Power enable from FT245BM
    si_wu_n   : out std_logic;            -- Send Immediately/Wake up to FT245BM.
    
    led       : out std_logic;            -- Output to single LED on BurchED FPGA board.
    
    led1      : out std_logic_vector(7 downto 0) -- Output to BurchED B5-LEDs module
    
    );
end chip;


architecture rtl of chip is


-----------------------------------------------------------------
--     Component declatations (not needed with Vhdl'93)
-----------------------------------------------------------------

component usb_if
port ( clk         : in std_logic;
      rst         : in std_logic;
      rxf_n       : in std_logic;
      txe_n       : in std_logic;
      data_in     : in std_logic_vector(7 downto 0);
      data_out    : out std_logic_vector(7 downto 0);
      dout_enable : out std_logic;
      wr          : out std_logic;
      rd_n        : out std_logic;
      pwren_n     : in std_logic;
      si_wu_n     : out std_logic;
      reg1        : out std_logic_vector(7 downto 0);
      reg2        : out std_logic_vector(7 downto 0);
      reg3        : out std_logic_vector(7 downto 0);
      reg4        : out std_logic_vector(7 downto 0);
      reg5        : out std_logic_vector(7 downto 0);
      reg6        : out std_logic_vector(7 downto 0);
      reg7        : out std_logic_vector(7 downto 0);
      reg8        : out std_logic_vector(7 downto 0);
      reg9        : out std_logic_vector(7 downto 0);
      reg10       : out std_logic_vector(7 downto 0);
      reg11       : out std_logic_vector(7 downto 0);
      reg12       : out std_logic_vector(7 downto 0);
      reg13       : out std_logic_vector(7 downto 0);
      reg14       : out std_logic_vector(7 downto 0);
      reg15       : out std_logic_vector(7 downto 0);
      reg16       : out std_logic_vector(7 downto 0);
      reg17       : out std_logic_vector(7 downto 0);
      reg18       : out std_logic_vector(7 downto 0);
      reg19       : out std_logic_vector(7 downto 0);
      reg20       : out std_logic_vector(7 downto 0);
      stat1       : in std_logic_vector(7 downto 0);
      stat2       : in std_logic_vector(7 downto 0);
      stat3       : in std_logic_vector(7 downto 0);
      stat4       : in std_logic_vector(7 downto 0);
      stat5       : in std_logic_vector(7 downto 0);
      stat6       : in std_logic_vector(7 downto 0);
      stat7       : in std_logic_vector(7 downto 0);
      stat8       : in std_logic_vector(7 downto 0);
      stat9       : in std_logic_vector(7 downto 0);
      stat10      : in std_logic_vector(7 downto 0);
      led1        : out std_logic_vector(7 downto 0);
      led2        : out std_logic_vector(7 downto 0)
);
end component;



-----------------------------------------------------------------
--     Signal declarations
-----------------------------------------------------------------

signal data_in     : std_logic_vector(7 downto 0);
signal data_out    : std_logic_vector(7 downto 0);
signal dout_enable : std_logic;

signal reg1        :  std_logic_vector(7 downto 0);
signal reg2        :  std_logic_vector(7 downto 0);
signal reg3        :  std_logic_vector(7 downto 0);
signal reg4        :  std_logic_vector(7 downto 0);
signal reg5        :  std_logic_vector(7 downto 0);
signal reg6        :  std_logic_vector(7 downto 0);
signal reg7        :  std_logic_vector(7 downto 0);
signal reg8        :  std_logic_vector(7 downto 0);
signal reg9        :  std_logic_vector(7 downto 0);
signal reg10       :  std_logic_vector(7 downto 0);
signal reg11       :  std_logic_vector(7 downto 0);
signal reg12       :  std_logic_vector(7 downto 0);
signal reg13       :  std_logic_vector(7 downto 0);
signal reg14       :  std_logic_vector(7 downto 0);
signal reg15       :  std_logic_vector(7 downto 0);
signal reg16       :  std_logic_vector(7 downto 0);
signal reg17       :  std_logic_vector(7 downto 0);
signal reg18       :  std_logic_vector(7 downto 0);
signal reg19       :  std_logic_vector(7 downto 0);
signal reg20       :  std_logic_vector(7 downto 0);
signal stat1       :  std_logic_vector(7 downto 0);
signal stat2       :  std_logic_vector(7 downto 0);
signal stat3       :  std_logic_vector(7 downto 0);
signal stat4       :  std_logic_vector(7 downto 0);
signal stat5       :  std_logic_vector(7 downto 0);
signal stat6       :  std_logic_vector(7 downto 0);
signal stat7       :  std_logic_vector(7 downto 0);
signal stat8       :  std_logic_vector(7 downto 0);
signal stat9       :  std_logic_vector(7 downto 0);
signal stat10      :  std_logic_vector(7 downto 0);
signal led2        :  std_logic_vector(7 downto 0);

signal count       :  unsigned(28 downto 0);

signal rst         :  std_logic;

begin

rst <= not(rst_n);

--
--  Top level tristate control
--

data <= data_out when dout_enable = '1' else "ZZZZZZZZ";
data_in <= data;

process(clk,rst)
begin
if(rst = '1') then 
  count <= (others => '0');
elsif(rising_edge(clk)) then
  count <= count + '1';
end if;
end process;


led <= count(24);


--
--  Combine output registers to produce status registers (and so they won't
--  be optimised out by the synthesis tools).
--


process(clk)
begin
if(rising_edge(clk)) then
  stat1 <= reg1 and reg11;
  stat2 <= reg2 and reg12;
  stat3 <= reg3 and reg13;
  stat4 <= reg4 and reg14;
  stat5 <= reg5 and reg15;
  stat6 <= reg6 and reg16;
  stat7 <= reg7 and reg17;
  stat8 <= reg8 and reg18;
  stat9 <= reg9 and reg19;
  stat10 <= reg10 and reg20;
end if;
end process;











 

-----------------------------------------------------------------
--   Component Instantiations 
-----------------------------------------------------------------


inst_usb_if:usb_if
port map( clk         => clk, -- in std_logic 
      rst         => rst, -- in std_logic 
      rxf_n       => rxf_n, -- in std_logic 
      txe_n       => txe_n, -- in std_logic 
      data_in     => data_in, -- in std_logic_vector(7 downto 0) 
      data_out    => data_out, -- out std_logic_vector(7 downto 0) 
      dout_enable => dout_enable, -- out std_logic 
      wr          => wr, -- out std_logic 
      rd_n        => rd_n, -- out std_logic 
      pwren_n     => pwren_n, -- in std_logic 
      si_wu_n     => si_wu_n, -- out std_logic 
      reg1        => reg1, -- out std_logic_vector(7 downto 0) 
      reg2        => reg2, -- out std_logic_vector(7 downto 0) 
      reg3        => reg3, -- out std_logic_vector(7 downto 0) 
      reg4        => reg4, -- out std_logic_vector(7 downto 0) 
      reg5        => reg5, -- out std_logic_vector(7 downto 0) 
      reg6        => reg6, -- out std_logic_vector(7 downto 0) 
      reg7        => reg7, -- out std_logic_vector(7 downto 0) 
      reg8        => reg8, -- out std_logic_vector(7 downto 0) 
      reg9        => reg9, -- out std_logic_vector(7 downto 0) 
      reg10       => reg10, -- out std_logic_vector(7 downto 0) 
      reg11       => reg11, -- out std_logic_vector(7 downto 0) 
      reg12       => reg12, -- out std_logic_vector(7 downto 0) 
      reg13       => reg13, -- out std_logic_vector(7 downto 0) 
      reg14       => reg14, -- out std_logic_vector(7 downto 0) 
      reg15       => reg15, -- out std_logic_vector(7 downto 0) 
      reg16       => reg16, -- out std_logic_vector(7 downto 0) 
      reg17       => reg17, -- out std_logic_vector(7 downto 0) 
      reg18       => reg18, -- out std_logic_vector(7 downto 0) 
      reg19       => reg19, -- out std_logic_vector(7 downto 0) 
      reg20       => reg20, -- out std_logic_vector(7 downto 0) 
      stat1       => stat1, -- in std_logic_vector(7 downto 0) 
      stat2       => stat2, -- in std_logic_vector(7 downto 0) 
      stat3       => stat3, -- in std_logic_vector(7 downto 0) 
      stat4       => stat4, -- in std_logic_vector(7 downto 0) 
      stat5       => stat5, -- in std_logic_vector(7 downto 0) 
      stat6       => stat6, -- in std_logic_vector(7 downto 0) 
      stat7       => stat7, -- in std_logic_vector(7 downto 0) 
      stat8       => stat8, -- in std_logic_vector(7 downto 0) 
      stat9       => stat9, -- in std_logic_vector(7 downto 0) 
      stat10      => stat10, -- in std_logic_vector(7 downto 0) 
      led1        => led1, -- out std_logic_vector(7 downto 0)
      led2        => led2
);


end rtl;
