//---------------------------------------------------------------------------
//
// USB Demo
//
// (c) Nial Stewart Developments Ltd 2002.
// www.nialstewart.co.uk
//
// This is a very simple demonstration to show how easily the registers in
// an FPGA based design can be accessed with the USB interface module based
// around the FTDI USB interface chip. FTDI provide a DLL which allows the
// target device to be directly driven without knowing having to know anything
// about USB.
// For more details of this device, including data sheets and a programmers
// guide see http://www.ftdichip.com
//
// FTDI provide an example application, but it's structure is fairly complex
// and difficult to understand. As a hardware engineer I just wanted the
// simplest way of accessing registers in a target FPGA possible, this
// is the result. This will only work if there's ONE USB interface board
// connected and this is hardwired to be Device 0 (other USB device types
// shouldn't affect operation).
//
// The different sections to note are
//
// 1) Opening the device (TForm1::Button1Click).
//
// 2) Simplest Write (TForm1::Button3Click).
//
// 3) Simplest Read (TForm1::Button2Click).
//
// 4) More elegant write that handles the inputs from all the
//    register input boxes (TForm1::Edit1KeyDown)
//
// 5) A timer triggered read that reads and updates the status register
//    outputs every 100ms (TForm1::Timer1Timer). (This could probably have
//    been done in a simple loop, but I couldn't work out how to loop
//    throught the output boxes.)
//---------------------------------------------------------------------------

#include "Unit1.h"

// This is Borland C++ builder stuff
#include <vcl.h>
#pragma hdrstop


#pragma package(smart_init)
#pragma resource "*.dfm"
TForm1 *Form1;

// The following are kept Global so all routines can see them.
#define NUMBER_OF_DEVICES   4
TFtDevice Devices[NUMBER_OF_DEVICES];
TFtDevice *CurDev = NULL;
FT_HANDLE ftHandle;

char connected = 0;
char holdoff = 0;


//---------------------------------------------------------------------------
__fastcall TForm1::TForm1(TComponent* Owner)
        : TForm(Owner)
{
}

//---------------------------------------------------------------------------
//
// Each device must be opened by the software before it can be accessed. This
// is a simple routine which is hardwired to attempt to open device 0 and
// report on the result.
//


void __fastcall TForm1::Button1Click(TObject *Sender)
{

// Only try to connect if did-connected
if(connected == 0){
    FT_STATUS ftStatus;

    // This is hardwired to open device 0 and
    // assign a handle to it.
    ftStatus = FT_Open(0,&ftHandle);
    if (!FT_SUCCESS(ftStatus)){
        Label22->Caption = "Can't Connect!";
        Label22->Color = clRed;
        connected = 0;
        }
    else{
        Label22->Caption = "Connected";
        Label22->Color = clLime;
        connected = 1;
    }
}

}

//---------------------------------------------------------------------------
//
// Simple routine to close the connection to the device. This happens
// automatically when the application is closed.

void __fastcall TForm1::Button5Click(TObject *Sender)
{
    connected = 0;
    FT_Close(ftHandle);
    Label22->Caption = "Not Connected";
    Label22->Color = clWhite;
}


//---------------------------------------------------------------------------
//
// Simplest register write function (called by the LED Write button).
//

void __fastcall TForm1::Button3Click(TObject *Sender)
{


char Buf[256];
ULONG nobytes;
ULONG bytesWritten;
char *data;

if(Edit31->Text == ""){
    return;
}
else{

    // Buf[] contains the data to be sent
    // First send a write command..
    Buf[0] = 0x0a;
    // ..then the address..
    Buf[1] = 0x15;
    // ..then the value.
    Buf[2] = char(StrToInt("0x"+ Edit31->Text));

    // Number of bytes to be transferred.
    nobytes = 3;

    data = Buf;

    // Call the FTDI DLL function.
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
}

}



//---------------------------------------------------------------------------
//
// Simple register read function (called by the LED Read button).
//
// This is slightly more complex than the register write function as we've
// got to send a 'read' instruction to the FPGA, then read the result.
//


void __fastcall TForm1::Button2Click(TObject *Sender)
{

// Variables needed to transmit a 'read' instruction.
char TxBuf[20];
ULONG nobytes;
ULONG bytesWritten;
char *data;

// Variable used to read the result.
FT_STATUS ftStatus;
DWORD EventDWord;
DWORD RxBytes;
DWORD TxBytes;
DWORD BytesReceived;
char RxBuf[256];
int Received;


// First send a read instruction
// 0x05 = 'read' instruction
TxBuf[0] = 0x05;
// 0x15 = Led register address
TxBuf[1] = 0x15;
nobytes = 2;
data = TxBuf;

FT_Write(ftHandle,data,nobytes,&bytesWritten);

// Now read and display what's received.

// Wait until something's received.
do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
} while (RxBytes < 1);

// Read the result (it goes into RxBuf, starting at RxBuf[0] each
// read.
ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);


Received = RxBuf[0];            // Cast to an integer
Received = Received &  0xFF;    // Mask off sign bits
// Display (Edit32 is the LED Read result box)
Edit32->Text = IntToHex(Received,2);

}


//---------------------------------------------------------------------------
//
// This is a slightly more elegant write routine than handles the
// register input edit boxes.
//
// Note that I've built this application so that the number of each input
// edit box corresponds to its associated register in the target design. This
// simplifies working out the target address in this demo, but in a real
// application this wouldn't be practical and it'd probably be easiest to
// deduce the target address using a switch statement against the sending
// edit box number or name (which are worked out below).


void __fastcall TForm1::Edit1KeyDown(TObject *Sender, WORD &Key,
      TShiftState Shift)
{

// Only send data on a return key
if (Key == VK_RETURN){

    // First find out who the sender is and deduce their 'address'.

    String Sentname;
    TEdit *Whosent = dynamic_cast<TEdit *>(Sender);
    Sentname = Whosent->Name;

    // This does an 'A'scii string 'TO' 'I'nt conversion jumping 4 characters
    // first.
    int Devnum = atoi((Sentname.c_str()) + 4);


    char Buf[256];
    ULONG nobytes;
    ULONG bytesWritten;
    char *data;

    if(Whosent->Text == ""){
        return;
    }
    else{

        nobytes = 3;

        Buf[0] = 0x0a;
        Buf[1] = Devnum;
        Buf[2] = char(StrToInt("0x"+ Whosent->Text));

        data = Buf;

        FT_Write(ftHandle,data,nobytes,&bytesWritten);

    }

}
}
//---------------------------------------------------------------------------

// The timer's set up to trigger this every 100mS.

void __fastcall TForm1::Timer1Timer(TObject *Sender)
{

// Only try a read if we're connected (else everything hangs).
if(connected == 1){
    FT_STATUS ftStatus;
    DWORD EventDWord;
    DWORD RxBytes;
    DWORD TxBytes;
    DWORD BytesReceived;
    int Received;
    char RxBuf[256];

    // Variables needed to transmit a 'read' instruction.
    char TxBuf[20];
    ULONG nobytes;
    ULONG bytesWritten;
    char *data;

    // Read all status registers.
    // First read's commented, the rest are a cut and
    // paste with the read address changed.


    TxBuf[0] = 0x05;   // Read instruction
    TxBuf[1] = 0x21;   // Address to read
    nobytes = 2;       // Number of bytes to send (rd and address)
    data = TxBuf;

    // First send a read instruction.
    FT_Write(ftHandle,data,nobytes,&bytesWritten);

    // Now read and display what's received...
    // ...wait until we've received the something...
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    // ...get it ...
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];            // Cast to an integer
    Received = Received &  0xFF;    // Mask off sign bits
    // ...and display it.
    Edit21->Text = IntToHex(Received,2);


    TxBuf[1] = 0x22;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;    
    Edit22->Text = IntToHex(Received,2);


    TxBuf[1] = 0x23;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;    
    Edit23->Text = IntToHex(Received,2);

    TxBuf[1] = 0x24;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;    
    Edit24->Text = IntToHex(Received,2);


    TxBuf[1] = 0x25;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;    
    Edit25->Text = IntToHex(Received,2);


    TxBuf[1] = 0x26;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;    
    Edit26->Text = IntToHex(Received,2);


    TxBuf[1] = 0x27;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;    
    Edit27->Text = IntToHex(Received,2);


    TxBuf[1] = 0x28;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;   
    Edit28->Text = IntToHex(Received,2);


    TxBuf[1] = 0x29;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;    
    Edit29->Text = IntToHex(Received,2);


    TxBuf[1] = 0x2A;
    data = TxBuf;
    FT_Write(ftHandle,data,nobytes,&bytesWritten);
    do {
        FT_GetStatus(ftHandle,&RxBytes,&TxBytes,&EventDWord);
    } while (RxBytes < 1);
    ftStatus = FT_Read(ftHandle,RxBuf,RxBytes,&BytesReceived);
    Received = RxBuf[0];
    Received = Received &  0xFF;
    Edit30->Text = IntToHex(Received,2);

}

return;

}

//---------------------------------------------------------------------------




