# ! /usr/bin/perl 
# tb_gen.pl

# Nial Stewart
# Nial Stewart Developments Ltd
# www.nialstewartdevelopments.co.uk
# May 2003.
#
#
# A Perl script to automatically generate a testbench structure
# around a given source VHDL file. This is quite basic with 
# only a bit of error checking. The script is fairly well
# commented and can be easily modified to suit usere preferences.
# The script should be run in the same directory as the souce
# file. The testbench will end up there too.
# The testbench generated is "tb_entity_name", this is easily
# changed if required.
#
# To run enter "Perl tb_gen.pl source_name.vhd"
#
# The top level entity is instantiated with and assigned 
# signal names which match the port names.The entity is declared
# which isn't necessary in VHDL '93, but has been kept in
# for those using '87 (and can easily be commented out).
#
# A search is done for reset and clock inputs, with processes
# generated depenging on user inputs as to reset polarity and
# delay and clock period. If non numerical answers are given 
# default values are used. NOTE: No initialisation of signals
# is done, this must be done manually for the reset and clock lines
# (and any other signals which need initialised.
# Note also that the comment headers for the reset and clkgen 
# processes are inserted whether there are any reset or clock
# processes or not.
#
# This script is for general use and no warranty is given as to 
# correct functionality.

  
use warnings;
use strict;

my @entity;
my $entity_name = "not_extracted_yet_zzz_huU";

my $bench_filename;
my $arch_name = "test"; # Architecture name.

my $tb_name;

my @ports;
my @port_array;
my $port_list;
my $last_port;
  
my $input_line;    

my @libraries;
my $library;
my $packages;

my @signals;

my $answer;

my @clocks;
my $clk_name;
my $clk_period;

my @resets;
my $rst_name;
my $rst_polarity;
my $inactive_polarity;
my $rst_period;

my $vhdl_file;
my $number;

my @source;
my $line;
my $text;


my $top_ports;

# VHDL file is passes as only argument.
$vhdl_file = shift;

$bench_filename = "tb_$vhdl_file\n";
#$bench_filename = "$vhdl_file_tb\n"; # An alternate bench name

open OP,"> $bench_filename" or die $1;
          
open IP,"$vhdl_file" or die $!;

# Read in entire vhdl source file.
@source = <IP>;

# Remove all lines starting with comments.
foreach (@source){
  s/((--)(.*))//g;
  }


# Detect libraries, entity name and ports.

#Need to monitor that we're in the port declarations of
#the main entity and not any sub components. Use $top_ports
#to do this
$top_ports = 0;

foreach (@source){
    if ($_ =~ /(\s*)(library)(\s*)(\w+)/i){ #If this line matches
        push @libraries , $4;               #push the library onto the
    }                                       #library list.

    
    if ($_ =~ /(\s*)(entity)(\s+)(\w+)/){
        $entity_name = $4;
        $top_ports = 1; # Flag we're in the top ports.
    }

    if ($_ =~ /(\s*)(end)(\s+)($entity_name)/){
        $top_ports = 0; # Have reached end of top ports.
    }

  #..port detected by presence of in, out, inout or buffer type..
    if ($_ =~ /(\W*)(\w+)(\s+)(:)(\s+)(in|out|inout|buffer)(\s+)(.*?)(;?)/){
    
        if($top_ports == 1){
            s/((;)(.*))//ig;      #Lose ; at end of lines.
            s/(port+?)(.*)/$2/ig; #Lose any port declaration
            s/(^\W*)(\(+?)(.*)/$3/ig;   #Lose any '(' at the start
            push @ports , $_;
        }
    }
  

}

# Detect any potential clocks and reset lines in the port list,
# save these in @clocks and @resets arrays.
foreach (@ports){
  if ($_ =~ /(clk|clock)/){
    push @clocks, $_;
  }
  
  if ($_ =~ /(rst|reset)/){
    push @resets, $_;
  }
}
 
# Print header to the benchfile
print OP "-----------------------------------------------------------------\n";
print OP "--\n";
print OP "--  tb_$entity_name\n";
print OP "--\n";
print OP "--\n";
print OP "--  Author  : \n";
print OP "--  Date    : \n";
print OP "--  Revision: \n";
print OP "--  Comments: Automatically generated from $entity_name \n";
print OP "--\n";
print OP "--\n";
print OP "-----------------------------------------------------------------\n";

print OP "\n\n";

# For each of the libraries used, look through the source file
# for the library declaration or a use line, and copy them to
# the bench file.
foreach $library (@libraries){
    foreach $line (@source){
        if($line =~ /(\s*)(library)(\s*)($library)(.*)(;)/i){
        print OP "$line";
        }
        if($line =~ /(\s*)(use)(\s+)($library)(.+)(;)/i){
        print OP "$line";
        }
    }
    print OP "\n";
    
}

print OP "\n";

# Print testbench entity name to file.
print OP "entity tb_$entity_name is\n";
print OP "end tb_$entity_name;\n";

print OP "\n\n";

print OP "architecture $arch_name of tb_$entity_name is\n";

print OP "\n\n";

print OP "-----------------------------------------------------------------\n";;
print OP "--     Component declatations (not needed for Vhdl'93)\n";
print OP "-----------------------------------------------------------------\n";

print OP "\n";

print OP "component $entity_name\n";
print OP "port (";

# Extract the last port from the port list as it's not
# terminated with a ';'
$last_port = pop(@ports);

# Print the rest
foreach $line (@ports){
    chomp $line;
    print OP "$line;\n";
    }

# Remove the carriage return then print the last port    
chomp $last_port;    
print OP "$last_port\n";
print OP ");\n";

# Add it back to the port list
push @ports,$last_port;


print OP "end component;\n";

print OP "\n\n\n";

print OP "-----------------------------------------------------------------\n";
print OP "--     Signal declarations\n";
print OP "-----------------------------------------------------------------\n";
print OP "\n";

# Unse the port list as a source of signal names and types.
@signals = @ports;


foreach (@signals){
  s/(\W*)(.+)(:)(\s*)(\w+)(\s*)(.+)/$2$3$4$7/i; #Remove the port direction
 print OP "signal $_;\n";
}

print OP "\n\n\n";
print OP "begin\n";

print OP "\n\n";

print "\n\n";

print OP "-----------------------------------------------------------------\n";
print OP "--          Reset Processes \n";
print OP "-----------------------------------------------------------------\n";

foreach (@resets){
  /(\s*)(\w+)(.*)/;
  $rst_name = $2;
  print "Is '$rst_name' a reset port? (y/n)  ";
  $answer = <STDIN>;
  if ($answer =~ /y/i){
    print "Is this active high or low? (h/l)  ";
    $answer = <STDIN>;
    if ($answer =~ /h/i){
      $rst_polarity = '1';
      $inactive_polarity = '0';
    }elsif ($answer =~ /l/i){
      $rst_polarity = '0'; 
      $inactive_polarity = '1';
    }else{
      print "Mistake with polarity input, defaults to active low\n";
      $rst_polarity = '0';
      $inactive_polarity = '1';
    }
    
    print "Active for ? ns  ";
    $answer = <STDIN>;
    chomp $answer;
    if ($answer eq ""){ # This line checks for an input 
      print "Mistake with period, defaults to 100 ns\n";
      $rst_period = 100;
    }
    elsif ($answer =~ /[^0-9]/){ # This detects non 0-9 characters
      print "Mistake with period, defaults to 100 ns\n";
      $rst_period = 100;
    }else{
      $rst_period = $answer;
    }

    print "\n\n";
    
    print OP "\n\n";
    print OP $rst_name,"_proc:process\n";
    print OP "begin\n";
    print OP "$rst_name <= '$rst_polarity';\n";
    print OP "wait for $rst_period NS;\n";
    print OP "$rst_name <= '$inactive_polarity';\n";
    print OP "wait;\n";
    print OP "end process;\n";
    print OP "\n\n";
     
  }
  else{ # No input from Is this a reset question.
  print "No \n";
  }
}  


print OP "\n\n";
print "\n\n";


print OP "-----------------------------------------------------------------\n";
print OP "--          Clk_gen Processes \n";
print OP "-----------------------------------------------------------------\n";

foreach (@clocks){
  /(\s*)(\w+)(.*)/;
  $clk_name = $2;
  print "Is $clk_name a clock? (y/n)  ";
  $answer = <STDIN>;
  if ($answer =~ /y/i){
    print "Period of this clock in nano-seconds?  ";    
    $answer = <STDIN>;
    chomp $answer;
    if ($answer eq ""){
      print "Mistake with period, defaults to 20 ns\n";
      $clk_period = 20;
    }
    elsif ($answer =~ /[^0-9]/){
      print "Mistake with period, defaults to 20 ns\n";
      $clk_period = 20;
    }else{
      $clk_period = $answer;
    }    
    
    
    $number = $clk_period/2;

    print "\n\n";

    print OP $clk_name,"_gen:process\n";
    print OP "begin\n";
    print OP "wait for $number NS;\n";
    print OP "$clk_name <= not $clk_name;\n";
    print OP "end process;\n";
    
    print OP "\n";
  }
  else{ # No input from Is this a clock question.
    print "No \n";
  }
}

print OP "\n\n\n\n";

print OP "-----------------------------------------------------------------\n";
print OP "--   Component Instantiations \n";
print OP "-----------------------------------------------------------------\n";

print OP "\n\n";
print OP "inst_",$entity_name,":",$entity_name,"\n";
print OP "port map(";

$last_port = pop(@ports);
foreach (@ports){
  s/(\W*)(\w+)(\s+)(:)(\s*)(\w*)/$1$2$3=> $2, --$5$6/i; # Substitutes ':' with '=> , --' and adds signame
  print OP "$_ \n";
}
  
$last_port =~ s/(\W*)(\w+)(\s+)(:)(\s*)(\w*)/$1$2$3=> $2 --$5$6/i; # Miss out the , for the last one
print OP "$last_port\n";
print OP ");\n\n\n";

print OP "end $arch_name;\n";


print "Testbench generated\n";


  